<?php

namespace App\Http\Controllers;

use App\Models\UploadApp;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class UploadAppController extends Controller
{
    private const DISK = 'public';
    private const DIR  = 'upload_apps';

    public function index()
    {
        $uploadApps = UploadApp::latest()->paginate(10);

        return view('admin.upload_apps.index', compact('uploadApps'));
    }

    public function create()
    {
        return view('admin.upload_apps.create');
    }

    public function store(Request $request)
    {
        $data = $this->validatedData($request, isUpdate: false);

        $data['apk_file'] = $this->storeApkFile($request);

        UploadApp::create($data);

        return redirect()
            ->route('upload-apps.index')
            ->with('success', 'APK uploaded successfully.');
    }

    public function edit(UploadApp $upload_app)
    {
        return view('admin.upload_apps.edit', [
            'uploadApp' => $upload_app,
        ]);
    }

    public function update(Request $request, UploadApp $upload_app)
    {
        $data = $this->validatedData($request, isUpdate: true);

        if ($request->hasFile('apk_file')) {
            $this->deleteApkIfExists($upload_app->apk_file);
            $data['apk_file'] = $this->storeApkFile($request);
        }

        $upload_app->update($data);

        return redirect()
            ->route('upload-apps.index')
            ->with('success', 'APK updated successfully.');
    }

    public function destroy(UploadApp $upload_app)
    {
        $this->deleteApkIfExists($upload_app->apk_file);

        $upload_app->delete();

        return redirect()
            ->route('upload-apps.index')
            ->with('success', 'Deleted successfully.');
    }

    public function download(UploadApp $upload_app)
    {
        if (!$upload_app->apk_file || !Storage::disk(self::DISK)->exists($upload_app->apk_file)) {
            return back()->with('error', 'File not found.');
        }

        return Storage::disk(self::DISK)->download(
            $upload_app->apk_file,
            basename($upload_app->apk_file)
        );
    }

    /**
     * -------------------------
     * Helpers
     * -------------------------
     */

    private function validatedData(Request $request, bool $isUpdate): array
    {
        return $request->validate([
            'app_name' => ['nullable', 'string', 'max:255'],
            'apk_file' => [$isUpdate ? 'nullable' : 'required', 'file'],
        ]);
    }

    private function storeApkFile(Request $request): string
    {
        $file = $request->file('apk_file');

        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $ext          = strtolower($file->getClientOriginalExtension());

        $safeName     = Str::slug($originalName) ?: 'app';
        $newFileName  = $safeName . '-' . time() . '.' . $ext;

        return $file->storeAs(self::DIR, $newFileName, self::DISK);
    }

    private function deleteApkIfExists(?string $path): void
    {
        if ($path && Storage::disk(self::DISK)->exists($path)) {
            Storage::disk(self::DISK)->delete($path);
        }
    }
}