<?php

namespace App\Http\Controllers;

use App\Models\SystemInfo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SystemInfoController extends Controller
{
    public function index()
    {
        $systemInfos = SystemInfo::latest()->paginate(10);
        return view('admin.system_infos.index', compact('systemInfos'));
    }

    public function create()
    {
        return view('admin.system_infos.create');
    }

    public function store(Request $request)
    {
        $data = $this->validatedData($request);

        // Upload files
        if ($request->hasFile('logo')) {
            $data['logo'] = $request->file('logo')->store('system_infos', 'public');
        }
        if ($request->hasFile('favicon')) {
            $data['favicon'] = $request->file('favicon')->store('system_infos', 'public');
        }

        SystemInfo::create($data);

        return redirect()->route('system-infos.index')->with('success', 'System info created successfully.');
    }

    public function show(SystemInfo $systemInfo)
    {
        return view('admin.system_infos.show', compact('systemInfo'));
    }

    public function edit(SystemInfo $systemInfo)
    {
        return view('admin.system_infos.edit', compact('systemInfo'));
    }

    public function update(Request $request, SystemInfo $systemInfo)
    {
        $data = $this->validatedData($request);

        // Replace logo
        if ($request->hasFile('logo')) {
            if ($systemInfo->logo && Storage::disk('public')->exists($systemInfo->logo)) {
                Storage::disk('public')->delete($systemInfo->logo);
            }
            $data['logo'] = $request->file('logo')->store('system_infos', 'public');
        }

        // Replace favicon
        if ($request->hasFile('favicon')) {
            if ($systemInfo->favicon && Storage::disk('public')->exists($systemInfo->favicon)) {
                Storage::disk('public')->delete($systemInfo->favicon);
            }
            $data['favicon'] = $request->file('favicon')->store('system_infos', 'public');
        }

        $systemInfo->update($data);

        return redirect()->route('system-infos.index')->with('success', 'System info updated successfully.');
    }

    public function destroy(SystemInfo $systemInfo)
    {
        // Delete files
        if ($systemInfo->logo && Storage::disk('public')->exists($systemInfo->logo)) {
            Storage::disk('public')->delete($systemInfo->logo);
        }
        if ($systemInfo->favicon && Storage::disk('public')->exists($systemInfo->favicon)) {
            Storage::disk('public')->delete($systemInfo->favicon);
        }

        $systemInfo->delete();

        return redirect()->route('system-infos.index')->with('success', 'System info deleted successfully.');
    }

    private function validatedData(Request $request): array
    {
        return $request->validate([
            'company_name'        => ['nullable', 'string', 'max:255'],
            'logo'                => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp,svg', 'max:2048'],
            'favicon'             => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp,ico,svg', 'max:1024'],
            'address'             => ['nullable', 'string', 'max:255'],
            'phone'               => ['nullable', 'string', 'max:255'],
            'whatsapp_number'     => ['nullable', 'string', 'max:255'],
            'footer_copey_right'  => ['nullable', 'string', 'max:255'],
        ]);
    }
}
